#ifndef IO_TNG_HEADER_H
#define IO_TNG_HEADER_H

/* Written by Weiguang Cui + AI, 01/11/2025 */

/**
 * \file io_tng_header.h
 *
 * Provides functions for reading and writing the header of TNG
 * files. TNG is a binary format based on HDF5, used by the AREPO code
 * for storing cosmological simulation data.
 */


/**********************************************************************\
 *    Includes                                                        * 
\**********************************************************************/
#include "io_tng_header_def.h"
#include "io_tng_def.h"
#include "io_logging.h"


/**********************************************************************\
 *    Global defines, structure definitions and typedefs              * 
\**********************************************************************/


/**********************************************************************\
 *    Prototypes of global functions                                  * 
\**********************************************************************/

/**
 * \brief Reads a TNG header.
 *
 * The function will first rewind the file pointer to the start
 * of the header and then read in everything. Then it will
 * position the file pointer to the start of the particle data.
 *
 * \param log  A logging object.
 * \param f    A TNG file object.
 *
 * \return A freshly filled header, or NULL, in case of memory
 *         problems.
 */
extern io_tng_header_t
io_tng_header_get(io_logging_t log, io_tng_t f);

/**
 * \brief This will delete a TNG header object.
 *
 * \param log      The logging object.
 * \param *header  A pointer to the variable holding the header object.
 *                 This variable will be set to NULL.
 *
 * \return Nothing.
 */
extern void
io_tng_header_del(io_logging_t log, io_tng_header_t *header);

/**
 * \brief Writes a header to the file.
 *
 * \param log     The logging object.
 * \param header  The header to write.
 * \param f       The file the header will be written to.
 *
 * \return Nothing.
 */
extern void
io_tng_header_write(io_logging_t log,
                       io_tng_header_t header,
                       io_tng_t f);

/**
 * \brief Writes the header information to the logfile.
 *
 * \param log     The logging object.
 * \param header  The header object.
 *
 * \return Nothing.
 */
extern void
io_tng_header_log(io_logging_t log, io_tng_header_t header);


#endif /* IO_TNG_HEADER_H */
