/* Written by Weiguang Cui + AI, 01/11/2025 */

/**
 * \file io_tng_header.c
 *
 * Provides functions for reading and writing the header of TNG files.
 * TNG is a binary format based on HDF5, used by the AREPO code
 * for storing cosmological simulation data.
 */

#ifdef WITH_HDF5

/**********************************************************************\
 *    Includes                                                        * 
\**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <stdint.h>

#include "../define.h"

#include "io_tng_header.h"
#include "io_util.h"
#include "io_file.h"


/**********************************************************************\
 *    Local defines, structure definitions and typedefs               * 
\**********************************************************************/
#define SKIP(f) {fseek(f, 4L, SEEK_CUR);}
int IO_FILE_TYPE = -1;  // Global variable to store file type

/**********************************************************************\
 *    Prototypes of local functions                                   * 
\**********************************************************************/


/**********************************************************************\
 *    Implementation of global functions                              * 
\**********************************************************************/
extern io_tng_header_t
io_tng_header_get(io_logging_t log, io_tng_t f)
{
	io_tng_header_t dummy;
	long skipsize;
        hid_t fpin, hdf5_headergrp, hdf5_attribute;;

#ifdef FOPENCLOSE
  //fprintf(stderr,"FOPENCLOSE: reading header information from %s ... ",f->fname);
  //fpin = fopen(f->fname,IO_FILE_MODE_READ);
    fpin = H5Fopen(f->fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    if (fpin < 0) {
        io_logging_fatal(log,"io_tng_header_get(): could not open file %s for reading",f->fname);
        return NULL;
    }
  //fprintf(stderr,"file successfully opened ");
#else
  fpin = f->file;
#endif
    
    IO_FILE_TYPE = f->ftype;

	/* Some sanity checks */
	if ((f == NULL) || (f->file < 0))
		return NULL;

	/* Check if there already is a header, do nothing then */
	if (f->header != NULL)
		return f->header;

	/* Create the header structure array */
	dummy = (io_tng_header_t)malloc((size_t)TNG_HEADER_SIZE+1); // make +1 larger because of trailing '\0' inserted by io_util_readstring()
	if (dummy == NULL) {
		io_logging_memfatal(log, "TNG header structure %d",TNG_HEADER_SIZE+1);
		return NULL;
	}
    
        hdf5_headergrp = H5Gopen(fpin, "/Header");

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "NumPart_ThisFile");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, dummy->np);
        H5Aclose(hdf5_attribute);

		// force to remove the particle type 3
		dummy->np[3] = 0;

		if(f->ftype == IO_FILE_TNG) // force to ignore the NumPart_Total attribute for TNG files
		{
			io_logging_msg(log, INT32_C(15), "TNG file: using NumPart_ThisFile instead of NumPart_Total");
			memcpy(dummy->nall, dummy->np, sizeof(dummy->np)); // Copy array contents
			memset(dummy->nallhighw, 0, sizeof(dummy->nallhighw));
		}
		else
		{
			hdf5_attribute = H5Aopen_name(hdf5_headergrp, "NumPart_Total");
			H5Aread(hdf5_attribute, H5T_NATIVE_UINT, dummy->nall);
			H5Aclose(hdf5_attribute);

			hdf5_attribute = H5Aopen_name(hdf5_headergrp, "NumPart_Total_HighWord");
			H5Aread(hdf5_attribute, H5T_NATIVE_UINT, dummy->nallhighw);
			H5Aclose(hdf5_attribute);
		}

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "MassTable");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, dummy->massarr);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Time");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->expansion);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "NumFilesPerSnapshot");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->numfiles);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_DoublePrecision");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagdoubleprecision);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Redshift");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->redshift);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "BoxSize");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->boxsize);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Omega0");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->omega0);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "OmegaLambda");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->omegalambda);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "HubbleParam");
        H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->hubbleparameter);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_Sfr");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagsfr);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_Feedback");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagfeedback);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_Cooling");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagcooling);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_StellarAge");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagstellarage);
        H5Aclose(hdf5_attribute);

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Flag_Metals");
        H5Aread(hdf5_attribute, H5T_NATIVE_INT, &dummy->flagmetals);
        H5Aclose(hdf5_attribute);

        /* Read TNG-specific attributes */
        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "UnitLength_in_cm");
        if (hdf5_attribute >= 0) {
            H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->UnitLength_in_cm);
            H5Aclose(hdf5_attribute);
        } else {
            dummy->UnitLength_in_cm = 3.08568e21;  /* Default to 1 kpc in cm */
        }

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "UnitMass_in_g");
        if (hdf5_attribute >= 0) {
            H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->UnitMass_in_g);
            H5Aclose(hdf5_attribute);
        } else {
            dummy->UnitMass_in_g = 1.989e43;  /* Default to 1e10 Msun in g */
        }

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "UnitVelocity_in_cm_per_s");
        if (hdf5_attribute >= 0) {
            H5Aread(hdf5_attribute, H5T_NATIVE_DOUBLE, &dummy->UnitVelocity_in_cm_per_s);
            H5Aclose(hdf5_attribute);
        } else {
            dummy->UnitVelocity_in_cm_per_s = 1.0e5;  /* Default to km/s in cm/s */
        }

        /* Initialize string attributes with empty strings */
        memset(dummy->Git_commit, 0, sizeof(dummy->Git_commit));
        memset(dummy->Git_date, 0, sizeof(dummy->Git_date));

        /* Try to read Git commit and date if they exist */
        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Git_commit");
        if (hdf5_attribute >= 0) {
            hid_t atype = H5Aget_type(hdf5_attribute);
            hid_t atype_mem = H5Tget_native_type(atype, H5T_DIR_ASCEND);
            H5Aread(hdf5_attribute, atype_mem, dummy->Git_commit);
            H5Tclose(atype_mem);
            H5Tclose(atype);
            H5Aclose(hdf5_attribute);
        }

        hdf5_attribute = H5Aopen_name(hdf5_headergrp, "Git_date");
        if (hdf5_attribute >= 0) {
            hid_t atype = H5Aget_type(hdf5_attribute);
            hid_t atype_mem = H5Tget_native_type(atype, H5T_DIR_ASCEND);
            H5Aread(hdf5_attribute, atype_mem, dummy->Git_date);
            H5Tclose(atype_mem);
            H5Tclose(atype);
            H5Aclose(hdf5_attribute);
        }

        H5Gclose(hdf5_headergrp);

/* @TODO: Rennehan : BEGIN BLOCK READING CAN REMOVE BELOW */
        //dummy->flagentropyu = 0;

	f->header = dummy;
  
#ifdef FOPENCLOSE
        H5Fclose(fpin);
        //fprintf(stderr,"and closed (temporarily)\n");
#endif

	return dummy;
}

extern void
io_tng_header_del(io_logging_t log, io_tng_header_t *header)
{
	if ( (header == NULL) || (*header == NULL) )
		return;

	free(*header);

	*header = NULL;

	return;
}

extern void
io_tng_header_write(io_logging_t log,
                       io_tng_header_t header,
                       io_tng_t f)
{
	if ( (header == NULL) || (f == NULL))
		return;

	if (f->mode != IO_FILE_WRITE)
		return;

	if (f->file < 0)
		return;

	if (header != f->header) {
		io_logging_msg(log, INT32_C(2), "This is a TNG file.");
		io_logging_msg(log, INT32_C(1),
		               "Writing a different header than stored in "
		               "the file object to the file.");
	}

	/* TODO: Write the header */

	return;
}


extern void
io_tng_header_log(io_logging_t log, io_tng_header_t header)
{
	io_logging_msg(log, INT32_C(5),
	               "Header object information:");
	io_logging_msg(log, INT32_C(5),
	               "  No of particles in file:       %" PRIi32,
	               header->np[0]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIi32,
	               header->np[1]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIi32,
	               header->np[2]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIi32,
	               header->np[3]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIi32,
	               header->np[4]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIi32,
	               header->np[5]);
	io_logging_msg(log, INT32_C(5),
	               "  Mass of particle species:      %e",
	               header->massarr[0]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %e",
	               header->massarr[1]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %e",
	               header->massarr[2]);

	/* TNG-specific attributes */
	if (header->Git_commit[0] != '\0') {
		io_logging_msg(log, INT32_C(5),
		               "  Git commit:                     %s",
		               header->Git_commit);
	}

	if (header->Git_date[0] != '\0') {
		io_logging_msg(log, INT32_C(5),
		               "  Git date:                       %s",
		               header->Git_date);
	}

	io_logging_msg(log, INT32_C(5),
	               "  UnitLength_in_cm:               %e",
	               header->UnitLength_in_cm);
	io_logging_msg(log, INT32_C(5),
	               "  UnitMass_in_g:                  %e",
	               header->UnitMass_in_g);
	io_logging_msg(log, INT32_C(5),
	               "  UnitVelocity_in_cm_per_s:       %e",
	               header->UnitVelocity_in_cm_per_s);

	io_logging_msg(log, INT32_C(5),
	               "                                 %e",
	               header->massarr[3]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %e",
	               header->massarr[4]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %e",
	               header->massarr[5]);
	io_logging_msg(log, INT32_C(5),
	               "  Expansion:                     %e",
	               header->expansion);
	io_logging_msg(log, INT32_C(5),
	               "  Redshift:                      %e",
	               header->redshift);
	io_logging_msg(log, INT32_C(5),
	               "  Flagsfr:                       %" PRIi32,
	               header->flagsfr);
	io_logging_msg(log, INT32_C(5),
	               "  Flag Feedback:                 %" PRIi32,
	               header->flagfeedback);
	if(IO_FILE_TYPE == IO_FILE_MTNG)
	{
		io_logging_msg(log, INT32_C(5), "For total number of particles, the code directly uses the sum of NumPart_ThisFile");
	}
	else
	{
		io_logging_msg(log, INT32_C(5),
		               "  No of particles in total:      %" PRIu32,
		               header->nall[0]);
		io_logging_msg(log, INT32_C(5),
		               "                                 %" PRIu32,
		               header->nall[1]);
		io_logging_msg(log, INT32_C(5),
		               "                                 %" PRIu32,
		               header->nall[2]);
		io_logging_msg(log, INT32_C(5),
		               "                                 %" PRIu32,
		               header->nall[3]);
		io_logging_msg(log, INT32_C(5),
		               "                                 %" PRIu32,
		               header->nall[4]);
		io_logging_msg(log, INT32_C(5),
		               "                                 %" PRIu32,
		               header->nall[5]);
	}

	io_logging_msg(log, INT32_C(5),
	               "  Flag Cooling:                  %" PRIi32,
	               header->flagcooling);
	io_logging_msg(log, INT32_C(5),
	               "  Number of files:               %" PRIi32,
	               header->numfiles);
	io_logging_msg(log, INT32_C(5),
	               "  Boxsize:                       %e",
	               header->boxsize);
	io_logging_msg(log, INT32_C(5),
	               "  Omega0:                        %e",
	               header->omega0);
	io_logging_msg(log, INT32_C(5),
	               "  OmegaLambda:                   %e",
	               header->omegalambda);
	io_logging_msg(log, INT32_C(5),
	               "  Hubble parameter:              %e",
	               header->hubbleparameter);
	io_logging_msg(log, INT32_C(5),
	               "  Flag Stellar Age:              %" PRIi32,
	               header->flagstellarage);
	io_logging_msg(log, INT32_C(5),
	               "  Flag Metals:                   %" PRIi32,
	               header->flagmetals);
	io_logging_msg(log, INT32_C(5),
	               "  No of particles in total (HW): %" PRIu32,
	               header->nallhighw[0]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIu32,
	               header->nallhighw[1]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIu32,
	               header->nallhighw[2]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIu32,
	               header->nallhighw[3]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIu32,
	               header->nallhighw[4]);
	io_logging_msg(log, INT32_C(5),
	               "                                 %" PRIu32,
	               header->nallhighw[5]);
	/*io_logging_msg(log, INT32_C(5),
	               "  Flag Entropy insted U:         %" PRIi32,
	               header->flagentropyu);*/

	return;
}


/**********************************************************************\
 *    Implementation of local functions                               * 
\**********************************************************************/

#endif // WITH_HDF5